<?php
/**
 * Affilka JSON API Handler
 * Used by: BC Game, Thrill, PlayBet, LTCCasino, Bitstarz, MetaSpins, Justbit, BetsIO, CyBet, Rakebit, Jackpotter, WhalePlay
 * 
 * EXACT match to original report.php logic including CPA qualification
 */

require_once __DIR__ . '/../includes/functions.php';

/**
 * Extract money value from deep nested structure - EXACT from original report.php
 */
function extractMoneyDeep($v): float {
    // Fast paths
    if (is_numeric($v)) return (float)$v;
    if (is_string($v)) {
        $s = preg_replace('/[^\d.\-]/', '', str_replace([' ', ','], ['', ''], $v));
        return is_numeric($s) ? (float)$s : 0.0;
    }
    if (!is_array($v)) return 0.0;

    // Known keys
    if (isset($v['amount_cents']) && is_numeric($v['amount_cents'])) return ((float)$v['amount_cents'])/100.0;
    if (isset($v['amount']) && is_numeric($v['amount'])) return (float)$v['amount'];
    if (isset($v['value'])) {
        $deep = extractMoneyDeep($v['value']);
        if ($deep !== 0.0) return $deep;
    }
    if (isset($v['original_amount'])) {
        $deep = extractMoneyDeep($v['original_amount']);
        if ($deep !== 0.0) return $deep;
    }

    // Heuristic: scan for numeric fields that *look* like money
    $best = 0.0;
    foreach ($v as $k => $vv) {
        $cand = extractMoneyDeep($vv);
        // prefer values coming from keys with "sum" or "amount"
        $keyScore = (stripos((string)$k,'sum')!==false || stripos((string)$k,'amount')!==false) ? 2 : 1;
        if ($cand * $keyScore > $best) $best = $cand * $keyScore;
    }
    return $best > 0 ? $best : 0.0;
}

/**
 * Get value by name from Affilka response row
 */
function valByName(array $row, string $name) {
    foreach ($row as $entry) {
        if (isset($entry['name']) && $entry['name'] === $name) return $entry['value'] ?? null;
    }
    return null;
}

/**
 * Try multiple metric names and return the first non-null value
 */
function valByAnyName(array $row, array $names) {
    foreach ($names as $n) {
        $v = valByName($row, $n);
        if ($v !== null) return $v;
    }
    return null;
}

/**
 * Fetch CPA-qualified FTD count from Affilka API
 * This uses group_by[]=player to get per-player first deposit sums
 * Only counts FTDs where first_deposits_sum >= threshold (bl_usd)
 * 
 * EXACT logic from original report.php fetchCPA_Affilka_Qualified()
 */
function fetchAffilkaCPAQualified(array $casino, string $startDate, string $endDate, bool $debug = false): array {
    $apiUrl = rtrim($casino['api_url'], '/');
    $apiKey = $casino['api_key'];
    $threshold = floatval($casino['bl_usd'] ?? 0);
    
    // If no threshold, no need to check qualification
    if ($threshold <= 0) {
        return ['ok' => false, 'qualifying_count' => 0, 'notes' => ['No BL threshold set']];
    }
    
    // Expanded alias set for brands that rename first-deposit sum metrics
    $sumAliases = [
        'first_deposits_sum',
        'first_deposits_sum_usd',
        'first_deposit_sum',
        'sum_first_deposits',
        'ftd_sum',
        'ftd_amount',
        'first_deposits_amount',
        'first_deposits_total',
        'first_deposits_total_usd',
        'first_deposit_amount',
        'first_deposit_amount_usd',
        'ftd_amount_usd',
        'ftd_total',
        'ftd_total_usd',
        'deposit_first_sum',
        'deposit_first_sum_usd',
    ];
    
    // Ask for multiple variants up-front; Affilka ignores unknown columns
    $columns = array_values(array_unique(array_merge(['first_deposits_count'], $sumAliases)));
    
    $queryCols = '';
    foreach ($columns as $c) {
        $queryCols .= '&columns[]=' . urlencode($c);
    }
    
    $apiBase = $apiUrl . '?from=' . urlencode($startDate) . '&to=' . urlencode($endDate) . '&conversion_currency=USD';
    
    // Grouped by player to get individual FTD sums
    $apiUrlGrouped = $apiBase . $queryCols . '&group_by[]=' . urlencode('player');
    // Totals only (fallback)
    $apiUrlTotals = $apiBase . $queryCols;
    
    $ret = [
        'ok' => false,
        'qualifying_count' => 0,
        'total_ftd_count' => 0,
        'total_ftd_sum' => 0.0,
        'notes' => $debug ? ["API (grouped): " . $apiUrlGrouped] : [],
    ];
    
    $headersBearer = [
        'Accept: application/json',
        'Content-Type: application/json',
        'Authorization: Bearer ' . $apiKey,
    ];
    $headersPlain = [
        'Accept: application/json',
        'Content-Type: application/json',
        'Authorization: ' . $apiKey,
    ];
    
    // Minimal curl JSON helper (kept local so we can handle HTTP 429 properly)
    $curlJson = function(string $url, array $headers): array {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 60,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4,
        ]);
        $body = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $errno = curl_errno($ch);
        $error = curl_error($ch);
        curl_close($ch);
        return ['body' => $body, 'code' => $code, 'errno' => $errno, 'error' => $error];
    };

    // --- grouped call with rate-limit backoff (EXACT behavior from report.php) ---
    $maxRetries = 3;
    $retryDelayMs = 1000;
    $responseBody = null;
    $httpCode = null;

    for ($retry = 0; $retry < $maxRetries; $retry++) {
        if ($retry > 0) {
            usleep($retryDelayMs * 1000);
            $retryDelayMs *= 2;
            if ($debug) $ret['notes'][] = "Retry attempt $retry after rate limiting";
        }

        $r = $curlJson($apiUrlGrouped, $headersBearer);
        $httpCode = $r['code'] ?? null;

        if (!empty($r['errno'])) {
            if ($debug) $ret['notes'][] = "cURL error ({$r['errno']}): {$r['error']}";
            continue;
        }

        if ($httpCode === 429) {
            if ($debug) $ret['notes'][] = "HTTP 429 - Rate limited. Retrying...";
            continue;
        }

        if ($httpCode === 401) {
            $r2 = $curlJson($apiUrlGrouped, $headersPlain);
            if (($r2['code'] ?? 0) < 400 && empty($r2['errno'])) {
                $responseBody = $r2['body'];
                $httpCode = $r2['code'];
                break;
            }
            if ($debug) $ret['notes'][] = "Retry failed (HTTP ".($r2['code'] ?? 'n/a')."): ".($r2['error'] ?? '');
            continue;
        }

        if (($httpCode ?? 0) >= 400) {
            if ($debug) $ret['notes'][] = "HTTP $httpCode on grouped CPA call: ".substr((string)$r['body'], 0, 200);
            break;
        }

        $responseBody = $r['body'];
        break;
    }

    $ret['http_code'] = $httpCode;
    if (($ret['http_code'] ?? 0) >= 400 || $responseBody === null) return $ret;

    $json = json_decode((string)$responseBody, true);
    if (!is_array($json) || !isset($json['rows']['data'])) {
        if ($debug) $ret['notes'][] = "Invalid JSON grouped: ".substr((string)$responseBody, 0, 200);
        return $ret;
    }

    $rows = $json['rows']['data'] ?? [];

    // Get totals from grouped response
    $totFtdCount_g = 0;
    $totFtdSum_g = 0.0;
    if (!empty($json['totals']['data'][0])) {
        foreach ($json['totals']['data'][0] as $e) {
            if (!isset($e['name'])) continue;
            if ($e['name'] === 'first_deposits_count') $totFtdCount_g = (int)$e['value'];
            if (in_array($e['name'], $sumAliases, true)) {
                $totFtdSum_g = max($totFtdSum_g, extractMoneyDeep($e['value']));
            }
        }
    }
    
    // Count qualifying FTDs (per-player)
    $qual = 0;
    $inconclusive = 0;
    foreach ($rows as $row) {
        $ftdCount = (int)(valByAnyName($row, ['first_deposits_count', 'ftd', 'ftd_count']) ?? 0);
        $sumVal = valByAnyName($row, $sumAliases);
        $ftdSum = extractMoneyDeep($sumVal);
        
        if ($ftdCount >= 1) {
            if ($ftdSum > 0) {
                if ($ftdSum >= $threshold) {
                    $qual++;
                }
            } else {
                $inconclusive++;
            }
        }
    }
    
    // Fetch totals-only as fallback (some brands have better data here)
    $responseT = httpRequest($apiUrlTotals, $headersBearer);
    if ($responseT['code'] === 401) {
        $responseT = httpRequest($apiUrlTotals, $headersPlain);
    }
    
    $totFtdCount_t = 0;
    $totFtdSum_t = 0.0;
    if ($responseT['success'] && !empty($responseT['body'])) {
        $jT = json_decode($responseT['body'], true);
        if (is_array($jT) && !empty($jT['totals']['data'][0])) {
            foreach ($jT['totals']['data'][0] as $e) {
                if (!isset($e['name'])) continue;
                if ($e['name'] === 'first_deposits_count') $totFtdCount_t = (int)$e['value'];
                if (in_array($e['name'], $sumAliases, true)) {
                    $totFtdSum_t = max($totFtdSum_t, extractMoneyDeep($e['value']));
                }
            }
        }
        if ($debug) $ret['notes'][] = "API (totals): " . $apiUrlTotals;
    }
    
    // Choose the better totals
    $totFtdCount = max($totFtdCount_g, $totFtdCount_t);
    $totFtdSum = max($totFtdSum_g, $totFtdSum_t);
    
    // If no per-player qualification yet, try totals-based fallback
    if ($qual === 0 && $totFtdCount > 0 && $threshold > 0) {
        // Fallback: min(total FTD count, floor(total FTD sum / BL))
        $bySumFloor = (int)floor($totFtdSum / $threshold);
        $fallbackQual = max(0, min($totFtdCount, $bySumFloor));
        if ($fallbackQual > 0) {
            $qual = $fallbackQual;
            if ($debug) $ret['notes'][] = "Fallback used: totals-based qualification. FTD={$totFtdCount}, Sum={$totFtdSum}, BL={$threshold}, Qualified={$qual}";
        } else {
            if ($debug) $ret['notes'][] = "Fallback still zero: FTD={$totFtdCount}, Sum={$totFtdSum}";
        }
    }
    
    $ret['ok'] = true;
    $ret['qualifying_count'] = $qual;
    $ret['total_ftd_count'] = $totFtdCount;
    $ret['total_ftd_sum'] = $totFtdSum;
    
    if ($debug) {
        $ret['notes'][] = "Per-player rows: " . count($rows) . ", Qualified: {$qual}, Inconclusive: {$inconclusive}";
    }
    
    return $ret;
}

/**
 * Fetch data from Affilka API - EXACT match to original report.php
 * Now includes proper CPA qualification check
 */
function fetchAffilkaData(array $casino, string $startDate, string $endDate, bool $debug = false): array {
    $apiUrl = rtrim($casino['api_url'], '/');
    $apiKey = $casino['api_key'];
    
    // Expanded alias set for FTD sum (same as report.php)
    $sumAliases = [
        'first_deposits_sum',
        'first_deposits_sum_usd',
        'first_deposit_sum',
        'sum_first_deposits',
        'ftd_sum',
        'ftd_amount',
        'first_deposits_amount',
        'first_deposits_total',
        'first_deposits_total_usd',
        'first_deposit_amount',
        'first_deposit_amount_usd',
        'ftd_amount_usd',
        'ftd_total',
        'ftd_total_usd',
        'deposit_first_sum',
        'deposit_first_sum_usd',
    ];
    
    // Build API URL for base metrics - include FTD sum for fallback calculation
    $url = $apiUrl . '?from=' . urlencode($startDate) 
         . '&to=' . urlencode($endDate) 
         . '&conversion_currency=USD'
         . '&columns[]=ngr'
         . '&columns[]=sb_ngr'
         . '&columns[]=first_deposits_count';
    
    // Add first FTD sum alias for fallback CPA calculation
    $url .= '&columns[]=first_deposits_sum';
    
    // Headers
    $headers = [
        'Accept: application/json',
        'Content-Type: application/json',
        'Authorization: Bearer ' . $apiKey
    ];
    
    $response = httpRequest($url, $headers);
    
    $result = [
        'casino' => $casino['name'],
        'api_type' => 'affilka',
        'success' => false,
        'ftds' => 0,
        'ngr' => 0,
        'cpa_bonus' => 0,
        'profit' => 0,
        'debug' => null
    ];
    
    if ($debug) {
        $result['debug'] = [
            'url' => $url,
            'http_code' => $response['code'],
            'raw_response' => $response['body'] ?? ''
        ];
    }
    
    // If Bearer auth failed with 401, try plain Authorization
    if ($response['code'] === 401) {
        $headers = [
            'Accept: application/json',
            'Content-Type: application/json',
            'Authorization: ' . $apiKey
        ];
        $response = httpRequest($url, $headers);
        
        if ($debug) {
            $result['debug']['retry_with_plain_auth'] = true;
            $result['debug']['retry_http_code'] = $response['code'];
            $result['debug']['retry_raw_response'] = substr($response['body'] ?? '', 0, 2000);
        }
    }
    
    if (!$response['success']) {
        $result['error'] = $response['error'] ?: 'HTTP ' . $response['code'];
        return $result;
    }
    
    $json = json_decode($response['body'], true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        $result['error'] = 'Invalid JSON response';
        return $result;
    }
    
    $result['success'] = true;
    
    // Parse response
    $currency = 'USD';
    $ftd = 0;
    $ngr = 0.0;
    $sb_ngr = 0.0;
    $ftd_sum = 0.0; // Total first deposits sum for CPA fallback
    
    // Try totals first (preferred path)
    if (!empty($json['totals']['data'][0])) {
        foreach ($json['totals']['data'][0] as $e) {
            if (!isset($e['name'])) continue;
            if ($e['name'] === 'currency') $currency = (string)$e['value'];
            if ($e['name'] === 'first_deposits_count') $ftd = (int)$e['value'];
            if ($e['name'] === 'ngr') $ngr = extractMoneyDeep($e['value']);
            if ($e['name'] === 'sb_ngr') $sb_ngr = extractMoneyDeep($e['value']);
            // Try to get FTD sum from any of the aliases
            if (in_array($e['name'], $sumAliases, true)) {
                $ftd_sum = max($ftd_sum, extractMoneyDeep($e['value']));
            }
        }
    } else {
        // Fallback to rows if no totals
        $rows = $json['rows']['data'] ?? [];
        foreach ($rows as $row) {
            $ftd += (int)(valByName($row, 'first_deposits_count') ?? 0);
            $ngr += extractMoneyDeep(valByName($row, 'ngr') ?? 0);
            $sb_ngr += extractMoneyDeep(valByName($row, 'sb_ngr') ?? 0);
            // Try to get FTD sum
            foreach ($sumAliases as $alias) {
                $sumVal = valByName($row, $alias);
                if ($sumVal !== null) {
                    $ftd_sum += extractMoneyDeep($sumVal);
                    break; // Only count once per row
                }
            }
            $cur = valByName($row, 'currency');
            if ($cur) $currency = (string)$cur;
        }
    }
    
    // Total NGR = ngr + sb_ngr (sportsbook NGR)
    $totalNgr = $ngr + $sb_ngr;
    
    // Calculate CPA bonus - WITH PROPER QUALIFICATION AND FALLBACK
    $cpaBonus = 0;
    $blThreshold = floatval($casino['bl_usd'] ?? 0);
    $cpaAmount = floatval($casino['cpa_usd'] ?? 0);
    $qualifiedFtds = 0;
    
    if ($cpaAmount > 0 && $ftd > 0) {
        if ($blThreshold > 0) {
            // Try per-player qualification first (may rate-limit)
            $cpaResult = fetchAffilkaCPAQualified($casino, $startDate, $endDate, $debug);
            
            if ($cpaResult['ok'] && $cpaResult['qualifying_count'] > 0) {
                $qualifiedFtds = $cpaResult['qualifying_count'];
                $cpaBonus = $qualifiedFtds * $cpaAmount;
                
                if ($debug) {
                    $result['debug']['cpa_qualification'] = $cpaResult;
                    $result['debug']['cpa_method'] = 'per-player (grouped API)';
                }
            } else {
                // Fallback EXACTLY like report.php:
                // qualified = min(FTD_count, floor(FTD_sum / BL))
                $bySumFloor = (int)floor($ftd_sum / $blThreshold);
                $qualifiedFtds = max(0, min($ftd, $bySumFloor));
                $cpaBonus = $qualifiedFtds * $cpaAmount;
                
                if ($debug) {
                    $result['debug']['cpa_qualification'] = $cpaResult;
                    $result['debug']['cpa_method'] = 'totals-based fallback (report.php)';
                    $result['debug']['cpa_fallback_calc'] = [
                        'ftd_count' => $ftd,
                        'ftd_sum' => $ftd_sum,
                        'baseline' => $blThreshold,
                        'floor_sum_div_bl' => $bySumFloor,
                        'qualified' => $qualifiedFtds,
                    ];
                }
            }
        } else {
            // No BL threshold, all FTDs qualify
            $qualifiedFtds = $ftd;
            $cpaBonus = $ftd * $cpaAmount;
        }
    }
    
    // Calculate profit: (NGR * Revenue Share %) + CPA Bonus
    // BUT: If NGR is negative, ignore negative rev-share (profit = CPA only)
    $revenueShare = floatval($casino['revenue_share'] ?? 100) / 100;
    $revShareProfit = $totalNgr * $revenueShare;
    
    // CPA/NGR rule from original: if total NGR negative, ignore negative rev-share
    if ($totalNgr < 0) {
        $profit = $cpaBonus; // Profit = CPA only when NGR is negative
    } else {
        $profit = $revShareProfit + $cpaBonus;
    }
    
    // Match report.php display:
    // - FTDs column shows TOTAL FTD count
    // - CPA is calculated from qualified FTDs (baseline)
    $result['ftds'] = $ftd;
    $result['ftds_raw'] = $ftd;
    $result['ftds_qualified'] = $qualifiedFtds;
    $result['ngr'] = round($totalNgr, 2);
    $result['cpa_bonus'] = round($cpaBonus, 2);
    $result['profit'] = round($profit, 2);
    
    return $result;
}
