<?php
/**
 * Afflines API Handler
 * Used by: Niyvi
 * 
 * EXACT match to original report.php logic
 * NOTE: Uses GET request with JSON body (unusual but required by API)
 */

require_once __DIR__ . '/../includes/functions.php';

/**
 * Fetch data from Afflines API - EXACT match to original report.php
 */
function fetchAfflinesData(array $casino, string $startDate, string $endDate, bool $debug = false): array {
    $apiUrl = $casino['api_url'];
    $apiKey = $casino['api_key'];
    
    // Get month string from date - EXACT from original
    $monthStr = substr($startDate, 0, 7); // Format: YYYY-MM
    
    // Request body - EXACT from original report.php
    $body = json_encode([
        'month' => $monthStr,
        'page' => 1
    ]);
    
    // Headers - EXACT from original
    $headers = [
        'Content-Type: application/json',
        'Authorization: Bearer ' . $apiKey
    ];
    
    // IMPORTANT: Afflines uses GET request with body (unusual)
    // This is handled by 'GET_WITH_BODY' method in httpRequest
    $response = httpRequest($apiUrl, $headers, 'GET_WITH_BODY', $body);
    
    $result = [
        'casino' => $casino['name'],
        'api_type' => 'afflines',
        'success' => false,
        'ftds' => 0,
        'ngr' => 0,
        'cpa_bonus' => 0,
        'profit' => 0,
        'debug' => null
    ];
    
    if ($debug) {
        $result['debug'] = [
            'url' => $apiUrl,
            'method' => 'GET with body',
            'month' => $monthStr,
            'http_code' => $response['code'],
            'raw_response' => substr($response['body'] ?? '', 0, 2000)
        ];
    }
    
    if (!$response['success']) {
        $result['error'] = $response['error'] ?: 'HTTP ' . $response['code'];
        return $result;
    }
    
    $json = json_decode($response['body'], true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        $result['error'] = 'Invalid JSON response';
        return $result;
    }
    
    $result['success'] = true;
    
    // Parse response - EXACT from original report.php
    // Look for overallStats object
    $ov = $json['overallStats'] ?? [];
    
    // FTD and earned values from overallStats
    $ftd = (int)($ov['total_ftd'] ?? 0);
    $earned = (float)($ov['earned'] ?? 0.0);
    
    // For Afflines, 'earned' is the profit directly
    $revenueShare = floatval($casino['revenue_share'] ?? 100) / 100;
    
    $result['ftds'] = $ftd;
    $result['ngr'] = round($earned, 2);  // earned = NGR equivalent
    $result['cpa_bonus'] = 0;
    $result['profit'] = round($earned * $revenueShare, 2);
    
    return $result;
}
