<?php
/**
 * Cellxpert XML API Handler
 * Used by: JackBit, CryptoLeo, Casinok, 2up
 * 
 * EXACT match to original report.php logic
 * CRITICAL: Cellxpert APIs ONLY work via IPv4 (handled in httpRequest)
 */

require_once __DIR__ . '/../includes/functions.php';

/**
 * Fetch data from Cellxpert XML API - EXACT match to original report.php
 */
function fetchCellxpertData(array $casino, string $startDate, string $endDate, bool $debug = false): array {
    $apiUrl = rtrim($casino['api_url'], '/');
    $apiKey = $casino['api_key'];
    $affiliateId = $casino['affiliate_id'] ?? '';
    
    // Build API URL with EXACT parameters from original report.php
    // IMPORTANT: URL format is {api_url}/?command=mediareport&fromdate={from}&todate={to}&DateFormat=month
    $queryParams = http_build_query([
        'command' => 'mediareport',
        'fromdate' => $startDate,
        'todate' => $endDate,
        'DateFormat' => 'month'
    ]);
    
    // EXACT from original: Ensure single slash before '?'
    $url = $apiUrl . '/?' . $queryParams;
    
    // Headers - EXACT from original report.php
    $headers = [
        'affiliateid: ' . $affiliateId,
        'x-api-key: ' . $apiKey
    ];
    
    $response = httpRequest($url, $headers);
    
    $result = [
        'casino' => $casino['name'],
        'api_type' => 'cellxpert',
        'success' => false,
        'ftds' => 0,
        'ngr' => 0,
        'cpa_bonus' => 0,
        'profit' => 0,
        'debug' => null
    ];
    
    if ($debug) {
        $result['debug'] = [
            'url' => preg_replace('/x-api-key:\s*[^\s]+/', 'x-api-key: ***', $url),
            'http_code' => $response['code'],
            'raw_response' => substr($response['body'] ?? '', 0, 2000)
        ];
    }
    
    if (!$response['success']) {
        $result['error'] = $response['error'] ?: 'HTTP ' . $response['code'];
        return $result;
    }
    
    // Parse XML response - EXACT from original report.php
    $xml = @simplexml_load_string((string)$response['body']);
    
    if ($xml === false) {
        $result['error'] = 'Invalid XML response';
        if ($debug) {
            $result['debug']['xml_parse_error'] = true;
        }
        return $result;
    }
    
    $result['success'] = true;
    
    $ftd = 0;
    $commission = 0.0;
    
    // Parse Cellxpert XML - EXACT logic from original report.php
    // Look for <row> elements with <FTD> and <Commission> tags
    if (isset($xml->row)) {
        foreach ($xml->row as $row) {
            $ftd += (int)$row->FTD;
            $commission += (float)$row->Commission;
        }
    }
    
    // For Cellxpert, Commission IS the profit (they report profit directly)
    // Revenue share is usually 100% since Commission already represents our share
    $revenueShare = floatval($casino['revenue_share'] ?? 100) / 100;
    
    $result['ftds'] = $ftd;
    $result['ngr'] = round($commission, 2);  // Commission = NGR equivalent
    $result['cpa_bonus'] = 0;  // Cellxpert doesn't use CPA model in original

    // Requirement: if Cellxpert profit is negative, show 0
    $profit = round($commission * $revenueShare, 2);
    $result['profit'] = max(0, $profit);

    return $result;
}
