<?php
/**
 * Cron Job Script
 * Run every 30 minutes to check profit changes (notifications)
 * and keep TODAY's profit snapshot up to date for charts.
 *
 * FOR CPANEL: Use the web URL with key parameter:
 * Command: /usr/bin/curl -s "https://yourdomain.com/dashboard/cron.php?key=YOUR_CRON_KEY" > /dev/null 2>&1
 * 
 * OR use PHP CLI with environment variable:
 * Command: CRON_KEY=YOUR_CRON_KEY /usr/bin/php /path/to/cron.php
 * 
 * Schedule: Every 30 minutes -> 0,30 * * * *
 */

// Load config first (needed for key validation)
require_once __DIR__ . '/config/config.php';

// Determine if running from CLI or Web
$isCLI = (php_sapi_name() === 'cli' || defined('STDIN'));

// Get the expected key from config
$expectedKey = defined('CRON_WEB_KEY') ? CRON_WEB_KEY : '00cddc2b205efa90f725a504b052e81a';

// Validate access
$authorized = false;

if ($isCLI) {
    // CLI mode: check for CRON_KEY environment variable OR allow if running as CLI without key
    $envKey = getenv('CRON_KEY');
    if ($envKey === $expectedKey || $expectedKey === 'YOUR_SECRET_CRON_KEY') {
        // Either key matches OR no key has been set (default value)
        $authorized = true;
    } else if (empty($envKey)) {
        // No env key provided - allow CLI access (for backward compatibility)
        $authorized = true;
    }
} else {
    // Web mode: require key parameter
    header('Content-Type: text/plain; charset=utf-8');
    
    // Stream output in browser
    @ini_set('output_buffering', 'off');
    @ini_set('zlib.output_compression', '0');
    while (ob_get_level() > 0) { @ob_end_flush(); }
    @ob_implicit_flush(true);
    
    $cronKey = $_GET['key'] ?? '';
    if ($cronKey === $expectedKey && $expectedKey !== 'YOUR_SECRET_CRON_KEY') {
        $authorized = true;
    }
}

if (!$authorized) {
    if (!$isCLI) {
        http_response_code(403);
    }
    die("Access denied.\nFor web access: cron.php?key=YOUR_KEY\nFor CLI: Set CRON_KEY environment variable or configure CRON_WEB_KEY in config.php\n");
}

// Load remaining dependencies
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/includes/casinos.php';
require_once __DIR__ . '/includes/pushover.php';

// Set execution time limit (5 minutes)
set_time_limit(300);

$logPrefix = "[" . date('Y-m-d H:i:s') . "] ";

echo $logPrefix . "Starting cron job...\n";

try {
    $db = getDB();
    
    // Fetch today's data from all casinos
    $today = date('Y-m-d');
    $startOfMonth = date('Y-m-01');
    
    echo $logPrefix . "Fetching data for $startOfMonth to $today\n";
    
    $data = fetchAllCasinosData($startOfMonth, $today);
    
    echo $logPrefix . "Fetched data from " . count($data['casinos']) . " casinos\n";
    echo $logPrefix . "Success: " . $data['totals']['success_count'] . ", Errors: " . $data['totals']['error_count'] . "\n";
    
    // Save daily snapshot for each casino
    $savedCount = 0;
    foreach ($data['casinos'] as $casino) {
        if ($casino['success'] && isset($casino['casino_id'])) {
            $saved = saveProfitHistory($today, $casino['casino_id'], [
                'ftds' => $casino['ftds'],
                'ngr' => $casino['ngr'],
                'cpa_bonus' => $casino['cpa_bonus'],
                'profit' => $casino['profit']
            ]);
            if ($saved) {
                $savedCount++;
            }
        }
    }
    
    echo $logPrefix . "Saved $savedCount profit history records\n";
    
    // Check for profit notification using TOTAL PROFIT from dashboard
    $totalProfit = $data['totals']['profit'];
    echo $logPrefix . "Current TOTAL profit: " . formatMoney($totalProfit) . "\n";
    
    $notificationResult = checkAndNotifyProfitChange($totalProfit);
    
    if ($notificationResult['notified']) {
        echo $logPrefix . "✅ Notification sent! Change: " . formatMoney($notificationResult['change']) . "\n";
    } else {
        echo $logPrefix . "ℹ️ No notification: " . ($notificationResult['reason'] ?? 'Unknown') . "\n";
    }
    
    // Log summary
    echo $logPrefix . "Summary:\n";
    echo $logPrefix . "  - Total Profit: " . formatMoney($data['totals']['profit']) . "\n";
    echo $logPrefix . "  - Total FTDs: " . $data['totals']['ftds'] . "\n";
    echo $logPrefix . "  - Total NGR: " . formatMoney($data['totals']['ngr']) . "\n";
    echo $logPrefix . "  - Total CPA: " . formatMoney($data['totals']['cpa_bonus']) . "\n";
    
    echo $logPrefix . "Cron job completed successfully!\n";
    
} catch (Exception $e) {
    echo $logPrefix . "ERROR: " . $e->getMessage() . "\n";
    echo $logPrefix . "Stack trace: " . $e->getTraceAsString() . "\n";
    exit(1);
}

exit(0);
