<?php
/**
 * Main Dashboard Page
 */

require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/includes/casinos.php';
require_once __DIR__ . '/includes/pushover.php';

requireAuth();

$user = getCurrentUser();
$csrfToken = generateCSRFToken();

// Get filter parameters
$filter = $_GET['filter'] ?? 'this_month';
$startDate = $_GET['start_date'] ?? null;
$endDate = $_GET['end_date'] ?? null;
$debug = isset($_GET['debug']) && $_GET['debug'] === '1';

// Use custom dates if provided, otherwise use filter preset
if ($startDate && $endDate) {
    $dateRange = ['start' => $startDate, 'end' => $endDate];
} else {
    $dateRange = getDateRange($filter);
}

// Fetch dashboard data
$summary = getDashboardSummary($dateRange['start'], $dateRange['end'], $debug);

// Check for profit notification
$notificationResult = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['check_notification'])) {
    if (validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $notificationResult = checkAndNotifyProfitChange($summary['current']['totals']['profit']);
    }
}

// Get chart data (last 30 days for trend)
$chartData = getProfitHistory(
    date('Y-m-d', strtotime('-30 days')),
    date('Y-m-d')
);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="#0f1923">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <title>Dashboard - Casino Reporting</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <!-- Mobile Menu Button (Modern Hamburger) -->
    <button class="mobile-menu-btn" onclick="toggleSidebar()" aria-label="Toggle menu">
        <span class="hamburger-line"></span>
        <span class="hamburger-line"></span>
        <span class="hamburger-line"></span>
    </button>
    
    <!-- Sidebar Overlay -->
    <div class="sidebar-overlay" onclick="closeSidebar()"></div>
    
    <div class="layout">
        <!-- Sidebar -->
        <aside class="sidebar" id="sidebar">
            <div class="sidebar-logo">
                <h1>🎰 Casino</h1>
                <span>Dashboard</span>
            </div>

            <nav class="sidebar-nav">
                <a href="dashboard.php" class="nav-link active">
                    <span class="nav-icon">📊</span>
                    Dashboard
                </a>
                <a href="settings.php" class="nav-link">
                    <span class="nav-icon">⚙️</span>
                    Settings
                </a>
                <a href="logout.php" class="nav-link">
                    <span class="nav-icon">🚪</span>
                    Logout
                </a>
            </nav>
            
            <div class="sidebar-footer">
                <p>Logged in as <strong><?= h($user['username']) ?></strong></p>
            </div>
        </aside>
        
        <!-- Main Content -->
        <main class="main-content">
            <header class="page-header">
                <div>
                    <h1>Dashboard</h1>
                    <p class="text-muted">Casino Affiliate Performance</p>
                </div>
                <div class="header-actions">
                    <form method="POST" style="display: inline;">
                        <input type="hidden" name="csrf_token" value="<?= h($csrfToken) ?>">
                        <input type="hidden" name="check_notification" value="1">
                        <button type="submit" class="btn btn-secondary btn-sm">
                            🔔 Check Notifications
                        </button>
                    </form>
                </div>
            </header>
            
            <?php if ($notificationResult): ?>
                <div class="alert <?= $notificationResult['notified'] ? 'alert-success' : 'alert-info' ?>">
                    <?php if ($notificationResult['notified']): ?>
                        ✅ Notification sent! Profit changed by <?= formatMoney($notificationResult['change']) ?>
                    <?php else: ?>
                        ℹ️ <?= h($notificationResult['reason'] ?? 'No notification needed') ?>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            
            <!-- Date Filters -->
            <div class="filters-bar">
                <div class="filter-buttons">
                    <a href="?filter=today" class="btn btn-filter <?= $filter === 'today' ? 'active' : '' ?>">Today</a>
                    <a href="?filter=yesterday" class="btn btn-filter <?= $filter === 'yesterday' ? 'active' : '' ?>">Yesterday</a>
                    <a href="?filter=last_7_days" class="btn btn-filter <?= $filter === 'last_7_days' ? 'active' : '' ?>">7 Days</a>
                    <a href="?filter=this_month" class="btn btn-filter <?= $filter === 'this_month' ? 'active' : '' ?>">This Month</a>
                    <a href="?filter=last_month" class="btn btn-filter <?= $filter === 'last_month' ? 'active' : '' ?>">Last Month</a>
                </div>
                
                <form method="GET" class="date-range-form">
                    <input type="date" name="start_date" value="<?= h($dateRange['start']) ?>" class="form-control form-control-sm">
                    <span class="text-muted">to</span>
                    <input type="date" name="end_date" value="<?= h($dateRange['end']) ?>" class="form-control form-control-sm">
                    <button type="submit" class="btn btn-primary btn-sm">Apply</button>
                    <label class="debug-toggle">
                        <input type="checkbox" name="debug" value="1" <?= $debug ? 'checked' : '' ?>>
                        Debug
                    </label>
                </form>
            </div>
            
            <!-- KPI Cards -->
            <div class="kpi-grid">
                <div class="kpi-card">
                    <div class="kpi-icon profit">💰</div>
                    <div class="kpi-content">
                        <p class="kpi-label">Total Profit</p>
                        <p class="kpi-value <?= $summary['current']['totals']['profit'] >= 0 ? 'text-profit' : 'text-loss' ?>">
                            <?= formatMoney($summary['current']['totals']['profit']) ?>
                        </p>
                        <p class="kpi-change <?= $summary['percentages']['profit'] >= 0 ? 'positive' : 'negative' ?>">
                            <?= $summary['percentages']['profit'] >= 0 ? '↑' : '↓' ?>
                            <?= abs($summary['percentages']['profit']) ?>% vs prev
                        </p>
                    </div>
                </div>
                
                <div class="kpi-card">
                    <div class="kpi-icon cpa">🎁</div>
                    <div class="kpi-content">
                        <p class="kpi-label">CPA Bonus</p>
                        <p class="kpi-value text-cpa"><?= formatMoney($summary['current']['totals']['cpa_bonus']) ?></p>
                        <p class="kpi-change <?= $summary['percentages']['cpa_bonus'] >= 0 ? 'positive' : 'negative' ?>">
                            <?= $summary['percentages']['cpa_bonus'] >= 0 ? '↑' : '↓' ?>
                            <?= abs($summary['percentages']['cpa_bonus']) ?>% vs prev
                        </p>
                    </div>
                </div>
                
                <div class="kpi-card">
                    <div class="kpi-icon ftd">👥</div>
                    <div class="kpi-content">
                        <p class="kpi-label">Total FTDs</p>
                        <p class="kpi-value"><?= number_format($summary['current']['totals']['ftds']) ?></p>
                        <p class="kpi-change <?= $summary['percentages']['ftds'] >= 0 ? 'positive' : 'negative' ?>">
                            <?= $summary['percentages']['ftds'] >= 0 ? '↑' : '↓' ?>
                            <?= abs($summary['percentages']['ftds']) ?>% vs prev
                        </p>
                    </div>
                </div>
                
                <div class="kpi-card">
                    <div class="kpi-icon casinos">🎰</div>
                    <div class="kpi-content">
                        <p class="kpi-label">Active Casinos</p>
                        <p class="kpi-value"><?= $summary['casino_count'] ?></p>
                        <p class="kpi-change neutral">
                            <?= $summary['current']['totals']['success_count'] ?> OK, 
                            <?= $summary['current']['totals']['error_count'] ?> err
                        </p>
                    </div>
                </div>
            </div>
            
            <!-- Chart Section -->
            <div class="card">
                <div class="card-header">
                    <h2>📈 Profit Trend (Last 30 Days)</h2>
                </div>
                <div class="card-body">
                    <div class="chart-container">
                        <canvas id="profitChart"></canvas>
                    </div>
                </div>
            </div>
            
            <!-- Casino Table -->
            <div class="card">
                <div class="card-header">
                    <h2>🎰 Casino Performance</h2>
                    <span class="text-muted"><?= formatDate($dateRange['start']) ?> - <?= formatDate($dateRange['end']) ?></span>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-cards">
                            <thead>
                                <tr>
                                    <th>Casino</th>
                                    <th>API Type</th>
                                    <th class="text-right">FTDs</th>
                                    <th class="text-right">NGR</th>
                                    <th class="text-right">CPA</th>
                                    <th class="text-right">Profit</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($summary['current']['casinos'] as $casino): ?>
                                    <tr>
                                        <td data-label="Casino">
                                            <strong><?= h($casino['casino']) ?></strong>
                                        </td>
                                        <td data-label="API Type">
                                            <span class="badge badge-<?= strtolower($casino['api_type']) ?>">
                                                <?= h(ucfirst($casino['api_type'])) ?>
                                            </span>
                                        </td>
                                        <td data-label="FTDs" class="text-right"><?= number_format($casino['ftds']) ?></td>
                                        <td data-label="NGR" class="text-right"><?= formatMoney($casino['ngr']) ?></td>
                                        <td data-label="CPA" class="text-right text-cpa"><?= formatMoney($casino['cpa_bonus']) ?></td>
                                        <td data-label="Profit" class="text-right">
                                            <?= formatMoneyWithClass($casino['profit']) ?>
                                        </td>
                                        <td data-label="Status">
                                            <?php if ($casino['success']): ?>
                                                <span class="status-badge success">✓ OK</span>
                                            <?php else: ?>
                                                <span class="status-badge error" title="<?= h($casino['error'] ?? 'Unknown error') ?>">
                                                    ✗ Error
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php if ($debug && isset($casino['debug'])): ?>
                                        <tr class="debug-row">
                                            <td colspan="7">
                                                <details>
                                                    <summary>Debug Info</summary>
                                                    <pre><?= h(json_encode($casino['debug'], JSON_PRETTY_PRINT)) ?></pre>
                                                </details>
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot>
                                <tr class="totals-row">
                                    <td colspan="2"><strong>TOTALS</strong></td>
                                    <td class="text-right"><strong><?= number_format($summary['current']['totals']['ftds']) ?></strong></td>
                                    <td class="text-right"><strong><?= formatMoney($summary['current']['totals']['ngr']) ?></strong></td>
                                    <td class="text-right text-cpa"><strong><?= formatMoney($summary['current']['totals']['cpa_bonus']) ?></strong></td>
                                    <td class="text-right">
                                        <strong><?= formatMoneyWithClass($summary['current']['totals']['profit']) ?></strong>
                                    </td>
                                    <td></td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
            
            <footer class="page-footer">
                <p>Last updated: <?= h($summary['current']['fetched_at']) ?></p>
            </footer>
        </main>
    </div>
    
    <script>
        // Mobile sidebar toggle
        function toggleSidebar() {
            const sidebar = document.getElementById('sidebar');
            const overlay = document.querySelector('.sidebar-overlay');
            const menuBtn = document.querySelector('.mobile-menu-btn');

            sidebar.classList.toggle('open');
            overlay.classList.toggle('active');
            menuBtn.classList.toggle('active');
            document.body.style.overflow = sidebar.classList.contains('open') ? 'hidden' : '';
        }

        function closeSidebar() {
            const sidebar = document.getElementById('sidebar');
            const overlay = document.querySelector('.sidebar-overlay');
            const menuBtn = document.querySelector('.mobile-menu-btn');

            sidebar.classList.remove('open');
            overlay.classList.remove('active');
            menuBtn.classList.remove('active');
            document.body.style.overflow = '';
        }

        // Close sidebar when a nav item is tapped (mobile)
        document.querySelectorAll('.sidebar-nav a').forEach((a) => {
            a.addEventListener('click', () => {
                if (window.innerWidth <= 1024) closeSidebar();
            });
        });

        // Close sidebar on window resize to desktop
        window.addEventListener('resize', function() {
            if (window.innerWidth > 1024) {
                closeSidebar();
            }
        });
        
        // Chart.js configuration
        const chartData = (<?= json_encode($chartData) ?> || []).map(r => ({
            date: r.date,
            profit: Number(r.profit || 0)
        }));

        const canvas = document.getElementById('profitChart');
        const container = canvas?.closest('.chart-container');

        if (!canvas || !container) {
            // nothing to render
        } else if (!chartData.length) {
            container.innerHTML = '<div class="chart-empty">No history yet — run <code>cron.php</code> once (then schedule it: <code>*/30 * * * *</code>) to start saving snapshots.</div>';
        } else {
            const ctx = canvas.getContext('2d');
            new Chart(ctx, {
                type: 'line',
                data: {
                    labels: chartData.map(d => d.date),
                    datasets: [{
                        label: 'Daily Profit',
                        data: chartData.map(d => d.profit),
                        borderColor: '#22c55e',
                        backgroundColor: 'rgba(34, 197, 94, 0.1)',
                        fill: true,
                        tension: 0.4,
                        pointRadius: window.innerWidth < 768 ? 2 : 4,
                        pointHoverRadius: 6,
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    interaction: {
                        intersect: false,
                        mode: 'index'
                    },
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            backgroundColor: '#1a2634',
                            titleColor: '#fff',
                            bodyColor: '#94a3b8',
                            borderColor: '#2d3a4d',
                            borderWidth: 1,
                            padding: 12,
                            displayColors: false,
                            callbacks: {
                                label: function(context) {
                                    return '$' + context.parsed.y.toLocaleString('en-US', { minimumFractionDigits: 2 });
                                }
                            }
                        }
                    },
                    scales: {
                        x: {
                            grid: {
                                color: 'rgba(255, 255, 255, 0.05)',
                                drawBorder: false
                            },
                            ticks: {
                                color: '#64748b',
                                maxRotation: 45,
                                minRotation: 45,
                                font: {
                                    size: window.innerWidth < 768 ? 10 : 12
                                }
                            }
                        },
                        y: {
                            grid: {
                                color: 'rgba(255, 255, 255, 0.05)',
                                drawBorder: false
                            },
                            ticks: {
                                color: '#64748b',
                                font: {
                                    size: window.innerWidth < 768 ? 10 : 12
                                },
                                callback: function(value) {
                                    return '$' + value.toLocaleString();
                                }
                            }
                        }
                    }
                }
            });
        }

    </script>
</body>
</html>
