<?php
/**
 * Casino Data Aggregator
 * Fetches data from all configured casinos
 */

require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/../api/affilka.php';
require_once __DIR__ . '/../api/cellxpert.php';
require_once __DIR__ . '/../api/afflines.php';

/**
 * Fetch data from a single casino based on its API type
 */
function fetchCasinoData(array $casino, string $startDate, string $endDate, bool $debug = false): array {
    switch (strtolower($casino['api_type'])) {
        case 'affilka':
            return fetchAffilkaData($casino, $startDate, $endDate, $debug);
        case 'cellxpert':
            return fetchCellxpertData($casino, $startDate, $endDate, $debug);
        case 'afflines':
            return fetchAfflinesData($casino, $startDate, $endDate, $debug);
        default:
            return [
                'casino' => $casino['name'],
                'api_type' => $casino['api_type'],
                'success' => false,
                'error' => 'Unknown API type: ' . $casino['api_type'],
                'ftds' => 0,
                'ngr' => 0,
                'cpa_bonus' => 0,
                'profit' => 0
            ];
    }
}

/**
 * Fetch data from all active casinos
 */
function fetchAllCasinosData(string $startDate, string $endDate, bool $debug = false): array {
    $casinos = getCasinos(true);
    $results = [];
    $totals = [
        'ftds' => 0,
        'ngr' => 0,
        'cpa_bonus' => 0,
        'profit' => 0,
        'success_count' => 0,
        'error_count' => 0
    ];
    
    foreach ($casinos as $casino) {
        $data = fetchCasinoData($casino, $startDate, $endDate, $debug);
        $data['casino_id'] = $casino['id'];
        $data['revenue_share'] = $casino['revenue_share'];
        $data['bl_usd'] = $casino['bl_usd'];
        $data['cpa_usd'] = $casino['cpa_usd'];
        
        $results[] = $data;
        
        if ($data['success']) {
            $totals['ftds'] += $data['ftds'];
            $totals['ngr'] += $data['ngr'];
            $totals['cpa_bonus'] += $data['cpa_bonus'];
            $totals['profit'] += $data['profit'];
            $totals['success_count']++;
        } else {
            $totals['error_count']++;
        }
    }

    // Sort casinos by highest profit first
    usort($results, function ($a, $b) {
        return ($b['profit'] ?? 0) <=> ($a['profit'] ?? 0);
    });
    
    // Round totals
    $totals['ngr'] = round($totals['ngr'], 2);
    $totals['cpa_bonus'] = round($totals['cpa_bonus'], 2);
    $totals['profit'] = round($totals['profit'], 2);

    return [
        'casinos' => $results,
        'totals' => $totals,
        'date_range' => [
            'start' => $startDate,
            'end' => $endDate
        ],
        'fetched_at' => date('Y-m-d H:i:s')
    ];
}

/**
 * Get dashboard summary with comparison to previous period
 */
function getDashboardSummary(string $startDate, string $endDate, bool $debug = false): array {
    // Current period data
    $current = fetchAllCasinosData($startDate, $endDate, $debug);
    
    // Calculate previous period for comparison
    $daysDiff = (strtotime($endDate) - strtotime($startDate)) / 86400 + 1;
    $prevEndDate = date('Y-m-d', strtotime($startDate . ' -1 day'));
    $prevStartDate = date('Y-m-d', strtotime($prevEndDate . ' -' . ($daysDiff - 1) . ' days'));
    
    // Get previous period from history (faster than API calls)
    $prevHistory = getProfitHistory($prevStartDate, $prevEndDate);
    $prevTotals = [
        'ftds' => 0,
        'ngr' => 0,
        'cpa_bonus' => 0,
        'profit' => 0
    ];
    
    foreach ($prevHistory as $row) {
        $prevTotals['ftds'] += $row['ftds'];
        $prevTotals['ngr'] += $row['ngr'];
        $prevTotals['cpa_bonus'] += $row['cpa_bonus'];
        $prevTotals['profit'] += $row['profit'];
    }
    
    // Calculate changes
    $changes = [
        'ftds' => $current['totals']['ftds'] - $prevTotals['ftds'],
        'ngr' => round($current['totals']['ngr'] - $prevTotals['ngr'], 2),
        'cpa_bonus' => round($current['totals']['cpa_bonus'] - $prevTotals['cpa_bonus'], 2),
        'profit' => round($current['totals']['profit'] - $prevTotals['profit'], 2)
    ];
    
    // Calculate percentages
    // NOTE: Profit can be negative, so we use abs(previous) as denominator.
    $percentages = [
        'ftds' => $prevTotals['ftds'] > 0 ? round(($changes['ftds'] / $prevTotals['ftds']) * 100, 1) : 0,
        'ngr' => $prevTotals['ngr'] > 0 ? round(($changes['ngr'] / $prevTotals['ngr']) * 100, 1) : 0,
        'cpa_bonus' => $prevTotals['cpa_bonus'] > 0 ? round(($changes['cpa_bonus'] / $prevTotals['cpa_bonus']) * 100, 1) : 0,
        'profit' => $prevTotals['profit'] != 0.0 ? round(($changes['profit'] / abs($prevTotals['profit'])) * 100, 1) : 0
    ];

    return [
        'current' => $current,
        'previous' => $prevTotals,
        'changes' => $changes,
        'percentages' => $percentages,
        'casino_count' => count($current['casinos'])
    ];
}
