<?php
/**
 * Helper Functions
 * EXACT match to original report.php curl logic
 */

require_once __DIR__ . '/../config/database.php';

/**
 * Get all active casinos
 */
function getCasinos(bool $activeOnly = true): array {
    $db = getDB();
    $sql = "SELECT * FROM casinos";
    if ($activeOnly) {
        $sql .= " WHERE is_active = 1";
    }
    $sql .= " ORDER BY name ASC";
    $stmt = $db->query($sql);
    return $stmt->fetchAll();
}

/**
 * Get single casino by ID
 */
function getCasino(int $id): ?array {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM casinos WHERE id = ?");
    $stmt->execute([$id]);
    $result = $stmt->fetch();
    return $result ?: null;
}

/**
 * Save casino (insert or update)
 */
function saveCasino(array $data): array {
    $db = getDB();
    
    $requiredFields = ['name', 'api_type', 'api_url', 'api_key'];
    foreach ($requiredFields as $field) {
        if (empty($data[$field])) {
            return ['success' => false, 'error' => "Field '$field' is required"];
        }
    }
    
    $fields = [
        'name' => $data['name'],
        'api_type' => $data['api_type'],
        'api_url' => $data['api_url'],
        'api_key' => $data['api_key'],
        'affiliate_id' => $data['affiliate_id'] ?? '',
        'revenue_share' => floatval($data['revenue_share'] ?? 100),
        'bl_usd' => floatval($data['bl_usd'] ?? 0),
        'cpa_usd' => floatval($data['cpa_usd'] ?? 0),
        'is_active' => isset($data['is_active']) ? 1 : 0
    ];
    
    try {
        if (!empty($data['id'])) {
            // Update
            $sql = "UPDATE casinos SET name = ?, api_type = ?, api_url = ?, api_key = ?, 
                    affiliate_id = ?, revenue_share = ?, bl_usd = ?, cpa_usd = ?, is_active = ?, 
                    updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($sql);
            $stmt->execute([
                $fields['name'], $fields['api_type'], $fields['api_url'], $fields['api_key'],
                $fields['affiliate_id'], $fields['revenue_share'], $fields['bl_usd'], 
                $fields['cpa_usd'], $fields['is_active'], $data['id']
            ]);
            return ['success' => true, 'id' => $data['id']];
        } else {
            // Insert
            $sql = "INSERT INTO casinos (name, api_type, api_url, api_key, affiliate_id, 
                    revenue_share, bl_usd, cpa_usd, is_active, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
            $stmt = $db->prepare($sql);
            $stmt->execute([
                $fields['name'], $fields['api_type'], $fields['api_url'], $fields['api_key'],
                $fields['affiliate_id'], $fields['revenue_share'], $fields['bl_usd'], 
                $fields['cpa_usd'], $fields['is_active']
            ]);
            return ['success' => true, 'id' => $db->lastInsertId()];
        }
    } catch (PDOException $e) {
        error_log("Casino save error: " . $e->getMessage());
        return ['success' => false, 'error' => 'Database error occurred'];
    }
}

/**
 * Delete casino
 */
function deleteCasino(int $id): bool {
    $db = getDB();
    $stmt = $db->prepare("DELETE FROM casinos WHERE id = ?");
    return $stmt->execute([$id]);
}

/**
 * Save profit history record
 */
function saveProfitHistory(string $date, int $casinoId, array $data): bool {
    $db = getDB();
    
    $sql = "INSERT INTO profit_history (date, casino_id, ftds, ngr, cpa_bonus, profit, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, NOW()) 
            ON DUPLICATE KEY UPDATE ftds = ?, ngr = ?, cpa_bonus = ?, profit = ?";
    
    $stmt = $db->prepare($sql);
    return $stmt->execute([
        $date, $casinoId, 
        $data['ftds'], $data['ngr'], $data['cpa_bonus'], $data['profit'],
        $data['ftds'], $data['ngr'], $data['cpa_bonus'], $data['profit']
    ]);
}

/**
 * Get profit history for charts
 */
function getProfitHistory(string $startDate, string $endDate, ?int $casinoId = null): array {
    $db = getDB();
    
    $sql = "SELECT date, SUM(ftds) as ftds, SUM(ngr) as ngr, SUM(cpa_bonus) as cpa_bonus, SUM(profit) as profit 
            FROM profit_history WHERE date BETWEEN ? AND ?";
    $params = [$startDate, $endDate];
    
    if ($casinoId) {
        $sql .= " AND casino_id = ?";
        $params[] = $casinoId;
    }
    
    $sql .= " GROUP BY date ORDER BY date ASC";
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll();
}

/**
 * Get total profit from last saved snapshot (returns null if never saved)
 */
function getLastSavedProfit(): ?float {
    $val = getSetting('last_profit_value', null);
    return $val !== null && $val !== '' ? floatval($val) : null;
}

/**
 * Make HTTP request with cURL - EXACT match to original report.php
 * Key: Always use IPv4, disable SSL verification for these APIs
 */
function httpRequest(string $url, array $headers = [], string $method = 'GET', ?string $body = null): array {
    $ch = curl_init();
    
    $opts = [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_SSL_VERIFYPEER => false,  // EXACT from original
        CURLOPT_SSL_VERIFYHOST => 0,       // EXACT from original
        CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4,  // CRITICAL: Force IPv4 for Cellxpert
        CURLOPT_USERAGENT => 'CasinoDashboard/1.0'
    ];
    
    if ($method === 'POST') {
        $opts[CURLOPT_POST] = true;
        if ($body !== null) {
            $opts[CURLOPT_POSTFIELDS] = $body;
        }
    } elseif ($method === 'GET_WITH_BODY' && $body !== null) {
        // Special case for Niyvi/Afflines - GET request with body
        $opts[CURLOPT_CUSTOMREQUEST] = 'GET';
        $opts[CURLOPT_POSTFIELDS] = $body;
    }
    
    curl_setopt_array($ch, $opts);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $errno = curl_errno($ch);
    $error = curl_error($ch);
    curl_close($ch);
    
    return [
        'success' => $httpCode >= 200 && $httpCode < 300,
        'code' => $httpCode,
        'body' => $response,
        'error' => $error,
        'errno' => $errno
    ];
}

/**
 * Format date for display
 */
function formatDate(string $date, string $format = 'M j, Y'): string {
    return date($format, strtotime($date));
}

/**
 * Get date range from filter
 */
function getDateRange(string $filter = 'this_month'): array {
    switch ($filter) {
        case 'today':
            $start = date('Y-m-d');
            $end = date('Y-m-d');
            break;
        case 'yesterday':
            $start = date('Y-m-d', strtotime('-1 day'));
            $end = date('Y-m-d', strtotime('-1 day'));
            break;
        case 'last_7_days':
            $start = date('Y-m-d', strtotime('-7 days'));
            $end = date('Y-m-d');
            break;
        case 'last_30_days':
            $start = date('Y-m-d', strtotime('-30 days'));
            $end = date('Y-m-d');
            break;
        case 'last_month':
            $start = date('Y-m-01', strtotime('last month'));
            $end = date('Y-m-t', strtotime('last month'));
            break;
        case 'this_month':
        default:
            $start = date('Y-m-01');
            $end = date('Y-m-d');
            break;
    }
    
    return ['start' => $start, 'end' => $end];
}
