<?php
/**
 * Pushover Notification Handler
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/functions.php';

/**
 * Send Pushover notification
 */
function sendPushover(string $title, string $message, int $priority = 0): array {
    $apiToken = getSetting('pushover_api_token', '');
    $userKey = getSetting('pushover_user_key', '');
    
    if (empty($apiToken) || empty($userKey)) {
        return ['success' => false, 'error' => 'Pushover credentials not configured'];
    }
    
    $data = [
        'token' => $apiToken,
        'user' => $userKey,
        'title' => $title,
        'message' => $message,
        'priority' => $priority,
        'sound' => $priority >= 1 ? 'cashregister' : 'pushover',
        'html' => 1
    ];
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => 'https://api.pushover.net/1/messages.json',
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($data),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return ['success' => false, 'error' => $error];
    }
    
    $result = json_decode($response, true);
    
    if ($httpCode === 200 && isset($result['status']) && $result['status'] === 1) {
        return ['success' => true];
    }
    
    return ['success' => false, 'error' => $result['errors'][0] ?? 'Unknown error'];
}

/**
 * Check profit change and send notification if threshold met
 */
function checkAndNotifyProfitChange(float $currentProfit): array {
    $isEnabled = getSetting('pushover_enabled', '0') === '1';
    
    // Get last saved profit - if never set, initialize it with current profit (first run)
    $lastProfitRaw = getSetting('last_profit_value', null);
    
    if ($lastProfitRaw === null || $lastProfitRaw === '') {
        // First run: save current profit as baseline, don't notify
        setSetting('last_profit_value', strval($currentProfit));
        return ['notified' => false, 'reason' => 'First run - initialized baseline profit to ' . formatMoney($currentProfit)];
    }
    
    $lastProfit = floatval($lastProfitRaw);
    
    if (!$isEnabled) {
        // Still update the last profit even if notifications disabled
        setSetting('last_profit_value', strval($currentProfit));
        return ['notified' => false, 'reason' => 'Notifications disabled (profit updated silently)'];
    }
    
    $threshold = floatval(getSetting('pushover_threshold', PUSHOVER_THRESHOLD));
    
    $change = $currentProfit - $lastProfit;
    $absChange = abs($change);
    
    if ($absChange < $threshold) {
        return [
            'notified' => false, 
            'reason' => "Change of " . formatMoney($change) . " below threshold of " . formatMoney($threshold),
            'change' => $change,
            'last_profit' => $lastProfit,
            'current_profit' => $currentProfit
        ];
    }
    
    // Build notification message
    $direction = $change >= 0 ? '📈 Profit UP!' : '📉 Profit DOWN!';
    $changeSign = $change >= 0 ? '+' : '';
    $emoji = $change >= 0 ? '🎰💰' : '⚠️';
    
    $title = "$emoji Casino Profit Alert";
    $message = "<b>$direction</b>\n\n" .
               "Current Total: <b>" . formatMoney($currentProfit) . "</b>\n" .
               "Change: <b>$changeSign" . formatMoney($change) . "</b>\n" .
               "Previous: " . formatMoney($lastProfit) . "\n\n" .
               date('M j, Y g:i A');
    
    $priority = $absChange >= ($threshold * 2) ? 1 : 0; // High priority for big changes
    
    $result = sendPushover($title, $message, $priority);
    
    if ($result['success']) {
        // Update last saved profit value
        setSetting('last_profit_value', strval($currentProfit));
        setSetting('last_notification_time', date('Y-m-d H:i:s'));
        
        return [
            'notified' => true,
            'change' => $change,
            'last_profit' => $lastProfit,
            'current_profit' => $currentProfit
        ];
    }
    
    return [
        'notified' => false,
        'reason' => 'Pushover send failed: ' . ($result['error'] ?? 'Unknown error'),
        'change' => $change
    ];
}

/**
 * Send test notification
 */
function sendTestNotification(): array {
    return sendPushover(
        '🎰 Test Notification',
        "Your Casino Dashboard notifications are working!\n\nThis is a test message.\n\n" . date('M j, Y g:i A'),
        0
    );
}
