<?php
/**
 * Settings Page
 */

require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/includes/pushover.php';

requireAuth();

$user = getCurrentUser();
$csrfToken = generateCSRFToken();

$message = '';
$messageType = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $message = 'Invalid request. Please try again.';
        $messageType = 'error';
    } else {
        $action = $_POST['action'] ?? '';
        
        switch ($action) {
            case 'save_casino':
                $result = saveCasino($_POST);
                if ($result['success']) {
                    $message = 'Casino saved successfully!';
                    $messageType = 'success';
                } else {
                    $message = $result['error'];
                    $messageType = 'error';
                }
                break;
                
            case 'delete_casino':
                $id = intval($_POST['casino_id'] ?? 0);
                if ($id && deleteCasino($id)) {
                    $message = 'Casino deleted successfully!';
                    $messageType = 'success';
                } else {
                    $message = 'Failed to delete casino.';
                    $messageType = 'error';
                }
                break;
                
            case 'save_pushover':
                setSetting('pushover_api_token', trim($_POST['pushover_api_token'] ?? ''));
                setSetting('pushover_user_key', trim($_POST['pushover_user_key'] ?? ''));
                setSetting('pushover_threshold', floatval($_POST['pushover_threshold'] ?? 100));
                setSetting('pushover_enabled', isset($_POST['pushover_enabled']) ? '1' : '0');
                $message = 'Pushover settings saved!';
                $messageType = 'success';
                break;
                
            case 'test_pushover':
                $result = sendTestNotification();
                if ($result['success']) {
                    $message = 'Test notification sent successfully!';
                    $messageType = 'success';
                } else {
                    $message = 'Failed to send test notification: ' . ($result['error'] ?? 'Unknown error');
                    $messageType = 'error';
                }
                break;
                
            case 'change_password':
                $result = changePassword(
                    $user['id'],
                    $_POST['current_password'] ?? '',
                    $_POST['new_password'] ?? ''
                );
                if ($result['success']) {
                    $message = 'Password changed successfully!';
                    $messageType = 'success';
                } else {
                    $message = $result['error'];
                    $messageType = 'error';
                }
                break;
                
            case 'reset_last_profit':
                setSetting('last_profit_value', '0');
                $message = 'Last profit value reset to $0. Next profit check will trigger a notification.';
                $messageType = 'success';
                break;
        }
    }
}

// Get data for display
$casinos = getCasinos(false);
$pushoverSettings = [
    'api_token' => getSetting('pushover_api_token', ''),
    'user_key' => getSetting('pushover_user_key', ''),
    'threshold' => getSetting('pushover_threshold', '100'),
    'enabled' => getSetting('pushover_enabled', '0') === '1',
    'last_profit' => getSetting('last_profit_value', '0'),
    'last_notification' => getSetting('last_notification_time', 'Never')
];

// Get casino for editing
$editCasino = null;
if (isset($_GET['edit'])) {
    $editCasino = getCasino(intval($_GET['edit']));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="#0f1923">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <title>Settings - Casino Dashboard</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <!-- Mobile Menu Button (Modern Hamburger) -->
    <button class="mobile-menu-btn" onclick="toggleSidebar()" aria-label="Toggle menu">
        <span class="hamburger-line"></span>
        <span class="hamburger-line"></span>
        <span class="hamburger-line"></span>
    </button>
    
    <!-- Sidebar Overlay -->
    <div class="sidebar-overlay" onclick="closeSidebar()"></div>
    
    <div class="layout">
        <!-- Sidebar -->
        <aside class="sidebar" id="sidebar">
            <div class="sidebar-logo">
                <h1>🎰 Casino</h1>
                <span>Dashboard</span>
            </div>

            <nav class="sidebar-nav">
                <a href="dashboard.php" class="nav-link">
                    <span class="nav-icon">📊</span>
                    Dashboard
                </a>
                <a href="settings.php" class="nav-link active">
                    <span class="nav-icon">⚙️</span>
                    Settings
                </a>
                <a href="logout.php" class="nav-link">
                    <span class="nav-icon">🚪</span>
                    Logout
                </a>
            </nav>
            
            <div class="sidebar-footer">
                <p>Logged in as <strong><?= h($user['username']) ?></strong></p>
            </div>
        </aside>
        
        <!-- Main Content -->
        <main class="main-content">
            <header class="page-header">
                <div>
                    <h1>Settings</h1>
                    <p class="text-muted">Manage casinos, notifications, and account</p>
                </div>
            </header>
            
            <?php if ($message): ?>
                <div class="alert alert-<?= $messageType ?>">
                    <?= h($message) ?>
                </div>
            <?php endif; ?>
            
            <!-- Casino Management -->
            <div class="card">
                <div class="card-header">
                    <h2>🎰 Casino API Configuration</h2>
                    <a href="settings.php" class="btn btn-primary btn-sm">+ Add Casino</a>
                </div>
                <div class="card-body">
                    <!-- Casino Form -->
                    <form method="POST" class="casino-form">
                        <input type="hidden" name="csrf_token" value="<?= h($csrfToken) ?>">
                        <input type="hidden" name="action" value="save_casino">
                        <?php if ($editCasino): ?>
                            <input type="hidden" name="id" value="<?= $editCasino['id'] ?>">
                        <?php endif; ?>
                        
                        <div class="form-grid">
                            <div class="form-group">
                                <label>Casino Name *</label>
                                <input type="text" name="name" class="form-control" required
                                    value="<?= h($editCasino['name'] ?? '') ?>"
                                    placeholder="e.g., BC Game">
                            </div>
                            
                            <div class="form-group">
                                <label>API Type *</label>
                                <select name="api_type" class="form-control" required>
                                    <option value="">Select API Type</option>
                                    <option value="affilka" <?= ($editCasino['api_type'] ?? '') === 'affilka' ? 'selected' : '' ?>>Affilka (JSON)</option>
                                    <option value="cellxpert" <?= ($editCasino['api_type'] ?? '') === 'cellxpert' ? 'selected' : '' ?>>Cellxpert (XML)</option>
                                    <option value="afflines" <?= ($editCasino['api_type'] ?? '') === 'afflines' ? 'selected' : '' ?>>Afflines</option>
                                </select>
                            </div>
                            
                            <div class="form-group form-group-full">
                                <label>API URL *</label>
                                <input type="url" name="api_url" class="form-control" required
                                    value="<?= h($editCasino['api_url'] ?? '') ?>"
                                    placeholder="https://api.example.com/stats">
                            </div>
                            
                            <div class="form-group">
                                <label>API Key *</label>
                                <input type="text" name="api_key" class="form-control" required
                                    value="<?= h($editCasino['api_key'] ?? '') ?>"
                                    placeholder="Your API key">
                            </div>
                            
                            <div class="form-group">
                                <label>Affiliate ID (Cellxpert)</label>
                                <input type="text" name="affiliate_id" class="form-control"
                                    value="<?= h($editCasino['affiliate_id'] ?? '') ?>"
                                    placeholder="Your affiliate ID">
                            </div>
                            
                            <div class="form-group">
                                <label>Revenue Share %</label>
                                <input type="number" name="revenue_share" class="form-control" step="0.01" min="0" max="100"
                                    value="<?= h($editCasino['revenue_share'] ?? '100') ?>"
                                    placeholder="100">
                            </div>
                            
                            <div class="form-group">
                                <label>BL Threshold (USD)</label>
                                <input type="number" name="bl_usd" class="form-control" step="0.01" min="0"
                                    value="<?= h($editCasino['bl_usd'] ?? '0') ?>"
                                    placeholder="0">
                            </div>
                            
                            <div class="form-group">
                                <label>CPA Amount (USD)</label>
                                <input type="number" name="cpa_usd" class="form-control" step="0.01" min="0"
                                    value="<?= h($editCasino['cpa_usd'] ?? '0') ?>"
                                    placeholder="0">
                            </div>
                            
                            <div class="form-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="is_active" value="1"
                                        <?= ($editCasino['is_active'] ?? true) ? 'checked' : '' ?>>
                                    Active
                                </label>
                            </div>
                        </div>
                        
                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">
                                <?= $editCasino ? 'Update Casino' : 'Add Casino' ?>
                            </button>
                            <?php if ($editCasino): ?>
                                <a href="settings.php" class="btn btn-secondary">Cancel</a>
                            <?php endif; ?>
                        </div>
                    </form>
                    
                    <!-- Existing Casinos Table -->
                    <?php if (!empty($casinos)): ?>
                        <h3 style="margin-top: 30px; color: var(--text-primary);">Configured Casinos</h3>
                        <div class="table-responsive">
                            <table class="table table-cards">
                                <thead>
                                    <tr>
                                        <th>Casino</th>
                                        <th>API Type</th>
                                        <th>Rev %</th>
                                        <th>BL</th>
                                        <th>CPA</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($casinos as $c): ?>
                                        <tr>
                                            <td data-label="Casino"><strong><?= h($c['name']) ?></strong></td>
                                            <td data-label="API Type"><span class="badge badge-<?= strtolower($c['api_type']) ?>"><?= h(ucfirst($c['api_type'])) ?></span></td>
                                            <td data-label="Rev %"><?= h($c['revenue_share']) ?>%</td>
                                            <td data-label="BL"><?= formatMoney($c['bl_usd']) ?></td>
                                            <td data-label="CPA"><?= formatMoney($c['cpa_usd']) ?></td>
                                            <td data-label="Status">
                                                <span class="status-badge <?= $c['is_active'] ? 'success' : 'inactive' ?>">
                                                    <?= $c['is_active'] ? 'Active' : 'Off' ?>
                                                </span>
                                            </td>
                                            <td data-label="Actions">
                                                <a href="?edit=<?= $c['id'] ?>" class="btn btn-sm btn-secondary">Edit</a>
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this casino?')">
                                                    <input type="hidden" name="csrf_token" value="<?= h($csrfToken) ?>">
                                                    <input type="hidden" name="action" value="delete_casino">
                                                    <input type="hidden" name="casino_id" value="<?= $c['id'] ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger">Del</button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Pushover Settings -->
            <div class="card">
                <div class="card-header">
                    <h2>🔔 Pushover Notifications</h2>
                </div>
                <div class="card-body">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?= h($csrfToken) ?>">
                        <input type="hidden" name="action" value="save_pushover">
                        
                        <div class="form-grid">
                            <div class="form-group">
                                <label>Pushover API Token</label>
                                <input type="text" name="pushover_api_token" class="form-control"
                                    value="<?= h($pushoverSettings['api_token']) ?>"
                                    placeholder="Your Pushover API token">
                            </div>
                            
                            <div class="form-group">
                                <label>Pushover User Key</label>
                                <input type="text" name="pushover_user_key" class="form-control"
                                    value="<?= h($pushoverSettings['user_key']) ?>"
                                    placeholder="Your Pushover user key">
                            </div>
                            
                            <div class="form-group">
                                <label>Notification Threshold (USD)</label>
                                <input type="number" name="pushover_threshold" class="form-control" step="1" min="1"
                                    value="<?= h($pushoverSettings['threshold']) ?>"
                                    placeholder="100">
                                <small class="form-help">Notify when profit changes by ±$X from last saved value</small>
                            </div>
                            
                            <div class="form-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="pushover_enabled" value="1"
                                        <?= $pushoverSettings['enabled'] ? 'checked' : '' ?>>
                                    Enable Notifications
                                </label>
                            </div>
                        </div>
                        
                        <div class="info-box">
                            <p><strong>Last Saved Profit:</strong> <?= formatMoney(floatval($pushoverSettings['last_profit'])) ?></p>
                            <p><strong>Last Notification:</strong> <?= h($pushoverSettings['last_notification']) ?></p>
                        </div>
                        
                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">Save Pushover Settings</button>
                            <button type="submit" name="action" value="test_pushover" class="btn btn-secondary">
                                Test Notification
                            </button>
                            <button type="submit" name="action" value="reset_last_profit" class="btn btn-warning"
                                onclick="return confirm('Reset last profit value to $0?')">
                                Reset Profit
                            </button>
                        </div>
                    </form>
                </div>
            </div>
            
            <!-- Change Password -->
            <div class="card">
                <div class="card-header">
                    <h2>🔒 Change Password</h2>
                </div>
                <div class="card-body">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?= h($csrfToken) ?>">
                        <input type="hidden" name="action" value="change_password">
                        
                        <div class="form-grid">
                            <div class="form-group">
                                <label>Current Password</label>
                                <input type="password" name="current_password" class="form-control" required
                                    placeholder="Enter current password">
                            </div>
                            
                            <div class="form-group">
                                <label>New Password</label>
                                <input type="password" name="new_password" class="form-control" required minlength="8"
                                    placeholder="Enter new password (min 8 chars)">
                            </div>
                        </div>
                        
                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">Change Password</button>
                        </div>
                    </form>
                </div>
            </div>
            
            <footer class="page-footer">
                <p>Casino Dashboard v1.0</p>
            </footer>
        </main>
    </div>
    
    <script>
        // Mobile sidebar toggle
        function toggleSidebar() {
            const sidebar = document.getElementById('sidebar');
            const overlay = document.querySelector('.sidebar-overlay');
            const menuBtn = document.querySelector('.mobile-menu-btn');

            sidebar.classList.toggle('open');
            overlay.classList.toggle('active');
            menuBtn.classList.toggle('active');
            document.body.style.overflow = sidebar.classList.contains('open') ? 'hidden' : '';
        }

        function closeSidebar() {
            const sidebar = document.getElementById('sidebar');
            const overlay = document.querySelector('.sidebar-overlay');
            const menuBtn = document.querySelector('.mobile-menu-btn');

            sidebar.classList.remove('open');
            overlay.classList.remove('active');
            menuBtn.classList.remove('active');
            document.body.style.overflow = '';
        }

        // Close sidebar when a nav item is tapped (mobile)
        document.querySelectorAll('.sidebar-nav a').forEach((a) => {
            a.addEventListener('click', () => {
                if (window.innerWidth <= 1024) closeSidebar();
            });
        });

        // Close sidebar on window resize to desktop
        window.addEventListener('resize', function() {
            if (window.innerWidth > 1024) {
                closeSidebar();
            }
        });
    </script>
</body>
</html>
